<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/Photo.php';
require_once __DIR__ . '/../middleware/jwtMiddleware.php';

class PhotoController {
    private $photoModel;
    
    public function __construct() {
        $database = new Database();
        $db = $database->getConnection();
        $this->photoModel = new Photo($db);
        error_log("✅ PhotoController initialized");
    }

    public function uploadPhoto() {
        $userId = jwtMiddleware();
        
        error_log("📸 Upload photo request for user: " . $userId);
        
        // Check if file was uploaded
        if (!isset($_FILES['photo']) || $_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['error' => 'No photo uploaded or upload error']);
            return;
        }

        $uploadedFile = $_FILES['photo'];
        
        // Security validations
        $maxFileSize = 10 * 1024 * 1024; // 10MB
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        
        // Check file size
        if ($uploadedFile['size'] > $maxFileSize) {
            http_response_code(400);
            echo json_encode(['error' => 'File too large. Maximum size is 10MB']);
            return;
        }
        
        // Check file type
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $uploadedFile['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed']);
            return;
        }
        
        // Check if user has reached photo limit
        $currentPhotoCount = $this->photoModel->getUserPhotoCount($userId);
        if ($currentPhotoCount >= 6) {
            http_response_code(400);
            echo json_encode(['error' => 'Maximum 6 photos allowed']);
            return;
        }

        try {
            $result = $this->photoModel->uploadUserPhoto($userId, $uploadedFile);
            
            if ($result['success']) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Photo uploaded successfully',
                    'photo' => $result['photo']
                ]);
            } else {
                http_response_code(500);
                echo json_encode(['error' => $result['error']]);
            }
        } catch (Exception $e) {
            error_log("❌ Photo upload error: " . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Photo upload failed']);
        }
    }

    public function setMainPhoto() {
        $userId = jwtMiddleware();
        $data = json_decode(file_get_contents("php://input"), true);
        
        if (empty($data['photo_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Photo ID is required']);
            return;
        }

        try {
            $success = $this->photoModel->setMainPhoto($userId, $data['photo_id']);
            
            if ($success) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Main photo updated successfully'
                ]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Failed to set main photo']);
            }
        } catch (Exception $e) {
            error_log("❌ Set main photo error: " . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Failed to set main photo']);
        }
    }

    public function deletePhoto() {
        $userId = jwtMiddleware();
        $data = json_decode(file_get_contents("php://input"), true);
        
        if (empty($data['photo_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Photo ID is required']);
            return;
        }

        try {
            $success = $this->photoModel->deleteUserPhoto($userId, $data['photo_id']);
            
            if ($success) {
                echo json_encode([
                    'success' => true,
                    'message' => 'Photo deleted successfully'
                ]);
            } else {
                http_response_code(400);
                echo json_encode(['error' => 'Failed to delete photo']);
            }
        } catch (Exception $e) {
            error_log("❌ Delete photo error: " . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Failed to delete photo']);
        }
    }

    public function getUserPhotos() {
        $userId = jwtMiddleware();
        
        try {
            $photos = $this->photoModel->getUserPhotos($userId);
            
            echo json_encode([
                'success' => true,
                'photos' => $photos
            ]);
        } catch (Exception $e) {
            error_log("❌ Get user photos error: " . $e->getMessage());
            http_response_code(500);
            echo json_encode(['error' => 'Failed to get user photos']);
        }
    }
}
?>