<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

class Photo {
    private $db;
    private $table_name = "user_photos";
    private $upload_dir;

    public function __construct($db) {
        $this->db = $db;
        $this->upload_dir = __DIR__ . '/../uploads/';
        
        // Create uploads directory if it doesn't exist
        if (!is_dir($this->upload_dir)) {
            mkdir($this->upload_dir, 0755, true);
        }
        
        error_log("✅ Photo model initialized");
    }

    public function uploadUserPhoto($userId, $uploadedFile) {
        try {
            // Generate unique filename for security
            $fileExtension = pathinfo($uploadedFile['name'], PATHINFO_EXTENSION);
            $safeExtension = strtolower($fileExtension);
            $uniqueFilename = uniqid() . '_' . bin2hex(random_bytes(8)) . '.' . $safeExtension;
            $filePath = $this->upload_dir . $uniqueFilename;
            
            // Move uploaded file
            if (!move_uploaded_file($uploadedFile['tmp_name'], $filePath)) {
                throw new Exception('Failed to move uploaded file');
            }
            
            // Check if this is the first photo (set as main)
            $photoCount = $this->getUserPhotoCount($userId);
            $isMain = $photoCount === 0;
            
            // Insert into database
            $query = "INSERT INTO " . $this->table_name . " 
                     (user_id, photo_url, photo_order, is_main) 
                     VALUES (:user_id, :photo_url, :photo_order, :is_main)";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':user_id', $userId);
            $stmt->bindParam(':photo_url', $uniqueFilename);
            $stmt->bindParam(':photo_order', $photoCount);
            $stmt->bindParam(':is_main', $isMain, PDO::PARAM_BOOL);
            
            if ($stmt->execute()) {
                $photoId = $this->db->lastInsertId();
                
                // Update user's photos count
                $this->updateUserPhotosCount($userId);
                
                return [
                    'success' => true,
                    'photo' => [
                        'id' => $photoId,
                        'photo_url' => $uniqueFilename,
                        'is_main' => $isMain,
                        'photo_order' => $photoCount
                    ]
                ];
            } else {
                // Clean up file if database insert failed
                if (file_exists($filePath)) {
                    unlink($filePath);
                }
                throw new Exception('Database insert failed');
            }
        } catch (Exception $e) {
            error_log("❌ Photo upload error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function setMainPhoto($userId, $photoId) {
        // First, set all user's photos to not main
        $query = "UPDATE " . $this->table_name . " 
                 SET is_main = FALSE 
                 WHERE user_id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        // Then set the specified photo as main
        $query = "UPDATE " . $this->table_name . " 
                 SET is_main = TRUE 
                 WHERE id = :photo_id AND user_id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':photo_id', $photoId);
        $stmt->bindParam(':user_id', $userId);
        
        return $stmt->execute() && $stmt->rowCount() > 0;
    }

    public function deleteUserPhoto($userId, $photoId) {
        // Get photo info before deletion
        $query = "SELECT photo_url, is_main FROM " . $this->table_name . " 
                 WHERE id = :photo_id AND user_id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':photo_id', $photoId);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        $photo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$photo) {
            return false;
        }
        
        // Delete from database
        $query = "DELETE FROM " . $this->table_name . " 
                 WHERE id = :photo_id AND user_id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':photo_id', $photoId);
        $stmt->bindParam(':user_id', $userId);
        
        if ($stmt->execute() && $stmt->rowCount() > 0) {
            // Delete physical file
            $filePath = $this->upload_dir . $photo['photo_url'];
            if (file_exists($filePath)) {
                unlink($filePath);
            }
            
            // Update user's photos count
            $this->updateUserPhotosCount($userId);
            
            // If deleted photo was main, set another photo as main
            if ($photo['is_main']) {
                $this->setFirstPhotoAsMain($userId);
            }
            
            return true;
        }
        
        return false;
    }

    public function getUserPhotos($userId) {
        $query = "SELECT id, photo_url, photo_order, is_main, created_at 
                 FROM " . $this->table_name . " 
                 WHERE user_id = :user_id 
                 ORDER BY is_main DESC, photo_order ASC";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUserPhotoCount($userId) {
        $query = "SELECT COUNT(*) as count FROM " . $this->table_name . " 
                 WHERE user_id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['count'];
    }

    private function updateUserPhotosCount($userId) {
        $count = $this->getUserPhotoCount($userId);
        
        $query = "UPDATE users SET photos_count = :count WHERE id = :user_id";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':count', $count);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
    }

    private function setFirstPhotoAsMain($userId) {
        $query = "SELECT id FROM " . $this->table_name . " 
                 WHERE user_id = :user_id 
                 ORDER BY photo_order ASC 
                 LIMIT 1";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();
        
        $photo = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($photo) {
            $this->setMainPhoto($userId, $photo['id']);
        }
    }
}
?>