<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';

class Profile {
    private $db;
    private $table_name = "users";

    public function __construct($db) {
        $this->db = $db;
        error_log("✅ Profile model initialized with table: {$this->table_name}");
    }

    public function updateUserProfile($user_id, $data) {
        try {
            $set_parts = [];
            $params = [':id' => $user_id];
            
            foreach ($data as $key => $value) {
                $set_parts[] = "`$key` = :$key";
                $params[":$key"] = $value;
            }
            
            $set_clause = implode(', ', $set_parts);
            $query = "UPDATE " . $this->table_name . " SET $set_clause WHERE id = :id";
            
            error_log("📝 Executing query: $query");
            error_log("📝 With params: " . print_r($params, true));
            
            $stmt = $this->db->prepare($query);
            
            foreach ($params as $param => $value) {
                $stmt->bindValue($param, $value);
            }
            
            $result = $stmt->execute();
            error_log("📝 Update result: " . ($result ? 'true' : 'false'));
            
            return $result;
            
        } catch (Exception $e) {
            error_log("❌ Database error in updateUserProfile: " . $e->getMessage());
            return false;
        }
    }

    public function updateUserPassword($user_id, $current_password, $new_password) {
        try {
            // First, get current password hash
            $query = "SELECT password FROM " . $this->table_name . " WHERE id = :id";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':id', $user_id);
            $stmt->execute();
            
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                error_log("❌ User not found for password update: $user_id");
                return false;
            }
            
            // Verify current password
            if (!password_verify($current_password, $user['password'])) {
                error_log("❌ Current password verification failed for user: $user_id");
                return false;
            }
            
            // Update to new password
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);
            $query = "UPDATE " . $this->table_name . " SET password = :password, updated_at = NOW() WHERE id = :id";
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':password', $new_password_hash);
            $stmt->bindParam(':id', $user_id);
            
            $result = $stmt->execute();
            error_log("✅ Password updated successfully for user: $user_id");
            
            return $result;
            
        } catch (Exception $e) {
            error_log("❌ Database error in updateUserPassword: " . $e->getMessage());
            return false;
        }
    }

    public function getUserById($user_id) {
        try {
            $query = "SELECT 
                id, name, email, nickname, gender, age, location, bio, preference, 
                profile_picture, avatar_url, created_at, updated_at, profile_complete 
              FROM " . $this->table_name . " 
              WHERE id = :id";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':id', $user_id);
            
            if ($stmt->execute()) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                error_log("✅ User data retrieved for: $user_id");
                return $user;
            }
            
            return false;
            
        } catch (Exception $e) {
            error_log("❌ Database error in getUserById: " . $e->getMessage());
            return false;
        }
    }
}
?>