<?php
declare(strict_types=1);

/**
 * Comprehensive logging utility for debugging API requests
 */

class Logger {
    private static $logFile;
    private static $enabled = true;
    
    public static function init(): void {
        $logDir = __DIR__ . '/../logs';
        
        // Create logs directory if it doesn't exist
        if (!is_dir($logDir)) {
            mkdir($logDir, 0777, true);
        }
        
        self::$logFile = $logDir . '/api_' . date('Y-m-d') . '.log';
        
        // Log server information
        self::log('🚀 Logger initialized', [
            'timestamp' => date('Y-m-d H:i:s'),
            'php_version' => PHP_VERSION,
            'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'N/A'
        ]);
    }
    
    public static function log(string $message, $data = null): void {
        if (!self::$enabled) {
            return;
        }
        
        $timestamp = date('Y-m-d H:i:s.u');
        $logEntry = "[$timestamp] $message\n";
        
        if ($data !== null) {
            if (is_array($data) || is_object($data)) {
                $logEntry .= self::formatData($data);
            } else {
                $logEntry .= "Data: " . $data . "\n";
            }
        }
        
        $logEntry .= "--------------------------------------------------\n";
        
        file_put_contents(self::$logFile, $logEntry, FILE_APPEND);
        
        // Also output to error_log for immediate visibility
        error_log($message);
    }
    
    private static function formatData($data): string {
        $output = "";
        
        if (is_array($data) || is_object($data)) {
            // Handle large objects by limiting depth
            $output = print_r($data, true);
            
            // Truncate if too large
            if (strlen($output) > 10000) {
                $output = substr($output, 0, 10000) . "\n... [TRUNCATED]";
            }
        } else {
            $output = (string)$data;
        }
        
        return $output . "\n";
    }
    
    public static function logRequest(): void {
        if (!self::$enabled) {
            return;
        }
        
        $requestId = uniqid('req_', true);
        
        self::log("📥 INCOMING REQUEST", [
            'request_id' => $requestId,
            'method' => $_SERVER['REQUEST_METHOD'] ?? 'N/A',
            'uri' => $_SERVER['REQUEST_URI'] ?? 'N/A',
            'query_string' => $_SERVER['QUERY_STRING'] ?? 'N/A',
            'full_url' => (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . 
                         ($_SERVER['HTTP_HOST'] ?? 'N/A') . 
                         ($_SERVER['REQUEST_URI'] ?? 'N/A'),
            'remote_addr' => $_SERVER['REMOTE_ADDR'] ?? 'N/A',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'N/A',
            'timestamp' => $_SERVER['REQUEST_TIME_FLOAT'] ?? microtime(true)
        ]);
        
        // Log GET parameters
        if (!empty($_GET)) {
            self::log("📋 GET Parameters", $_GET);
        }
        
        // Log POST data (if any)
        $postData = [];
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // For form data
            if (!empty($_POST)) {
                $postData['form'] = $_POST;
            }
            
            // For JSON input
            $jsonInput = file_get_contents('php://input');
            if ($jsonInput) {
                $jsonData = json_decode($jsonInput, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $postData['json'] = $jsonData;
                } else {
                    $postData['raw'] = $jsonInput;
                }
            }
            
            if (!empty($postData)) {
                self::log("📦 POST Data", $postData);
            }
        }
        
        // Log headers
        $headers = [];
        foreach ($_SERVER as $key => $value) {
            if (strpos($key, 'HTTP_') === 0) {
                $header = str_replace('_', ' ', substr($key, 5));
                $header = ucwords(strtolower($header));
                $header = str_replace(' ', '-', $header);
                $headers[$header] = $value;
            }
        }
        
        if (!empty($headers)) {
            self::log("📄 Request Headers", $headers);
        }
        
        // Store request ID for tracking
        define('REQUEST_ID', $requestId);
        $_SERVER['REQUEST_ID'] = $requestId;
    }
    
    public static function logResponse($response): void {
        if (!self::$enabled) {
            return;
        }
        
        $responseData = is_array($response) ? $response : ['response' => $response];
        
        self::log("📤 OUTGOING RESPONSE", [
            'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
            'http_code' => http_response_code(),
            'response' => $responseData
        ]);
    }
    
    public static function logError(string $error, \Throwable $exception = null): void {
        if (!self::$enabled) {
            return;
        }
        
        $errorData = [
            'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
            'error' => $error,
            'file' => $exception ? $exception->getFile() : 'N/A',
            'line' => $exception ? $exception->getLine() : 'N/A',
            'trace' => $exception ? $exception->getTraceAsString() : 'N/A'
        ];
        
        self::log("❌ ERROR OCCURRED", $errorData);
    }
    
    public static function logMiddleware(string $message, $data = null): void {
        if (!self::$enabled) {
            return;
        }
        
        self::log("🔐 MIDDLEWARE: $message", [
            'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
            'data' => $data
        ]);
    }
    
    public static function logDatabase(string $query, $params = null, $executionTime = null): void {
        if (!self::$enabled) {
            return;
        }
        
        $logData = [
            'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
            'query' => $query
        ];
        
        if ($params) {
            $logData['params'] = $params;
        }
        
        if ($executionTime) {
            $logData['execution_time_ms'] = $executionTime * 1000;
        }
        
        self::log("🗄️ DATABASE QUERY", $logData);
    }
    
    public static function enable(bool $enabled = true): void {
        self::$enabled = $enabled;
    }
    
    public static function getLogFile(): string {
        return self::$logFile;
    }
}

// Initialize logger
Logger::init();
Logger::logRequest();