<?php
declare(strict_types=1);

require_once __DIR__ . '/../vendor/autoload.php';
require_once __DIR__ . '/../config/env.php';

use Firebase\JWT\JWT;
use Firebase\JWT\Key;

function jwt_error(int $code, string $message): void {
    if (!headers_sent()) {
        header('Content-Type: application/json');
    }
    http_response_code($code);
    echo json_encode(['error' => $message]);
    exit();
}

function get_authorization_header(): ?string {
    if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
        return trim($_SERVER['HTTP_AUTHORIZATION']);
    }

    if (function_exists('apache_request_headers')) {
        $headers = apache_request_headers();
        foreach ($headers as $k => $v) {
            if (strcasecmp($k, 'Authorization') === 0) {
                return trim($v);
            }
        }
    }

    return null;
}

function jwtMiddleware(): int {
    // Skip middleware for public actions
    $action = $_GET['action'] ?? '';
    $publicActions = ['login', 'signup', 'logout'];
    
    if (in_array($action, $publicActions)) {
        return 0;
    }

    // Try Authorization header first
    $jwt = null;
    $authHeader = get_authorization_header();
    if ($authHeader && stripos($authHeader, 'Bearer ') === 0) {
        $jwt = substr($authHeader, 7);
    }

    if (!$jwt) {
        jwt_error(401, 'Authentication required');
    }

    // Load secret from environment
    $secret_key = $_ENV['JWT_SECRET'] ?? getenv('JWT_SECRET');
    if (!$secret_key) {
        jwt_error(500, 'Server configuration error');
    }

    try {
        if (property_exists(JWT::class, 'leeway')) {
            JWT::$leeway = 30;
        }

        $decoded = JWT::decode($jwt, new Key($secret_key, 'HS256'));

        $user_id = $decoded->user_id ?? null;

        if (!$user_id) {
            jwt_error(401, 'Invalid token payload');
        }

        return (int)$user_id;

    } catch (\Firebase\JWT\ExpiredException $e) {
        jwt_error(401, 'Token expired');
    } catch (\Firebase\JWT\SignatureInvalidException $e) {
        jwt_error(401, 'Invalid token signature');
    } catch (\Throwable $e) {
        jwt_error(401, 'Invalid token');
    }
}
?>